#include <avr/io.h>
#include "hd61830.h"

#define E_HIGH  	sbi(HD61830_CONTROL_PORT, HD61830_E)
#define E_LOW    	cbi(HD61830_CONTROL_PORT, HD61830_E)
#define RS_HIGH 	sbi(HD61830_CONTROL_PORT, HD61830_RS)
#define RS_LOW   	cbi(HD61830_CONTROL_PORT, HD61830_RS)
#define RW_HIGH 	sbi(HD61830_CONTROL_PORT, HD61830_RW)
#define RW_LOW   	cbi(HD61830_CONTROL_PORT, HD61830_RW)
#define CS_HIGH 	sbi(HD61830_CONTROL_PORT, HD61830_CS)
#define CS_LOW   	cbi(HD61830_CONTROL_PORT, HD61830_CS)
#define DATA_INPUT 	(HD61830_DATA_DDR = 0)
#define DATA_OUTPUT	(HD61830_DATA_DDR = 0xff)

// registers
#define MODE	0x00
#define CHAR_PITCH	0x01
#define NUM_OF_CHAR	0x02
#define NUM_OF_TIME_DIVISIONS	0x03
#define CURSOR_POS	0x04
#define START_LOW_ADDR	0x08
#define START_HIGH_ADDR	0x09
#define LOW_ADDR		0x0A
#define HIGH_ADDR		0x0B
#define WRITE_DATA		0x0C
#define READ_DATA		0x0D
#define CLEAR_BIT		0x0E
#define SET_BIT			0x0F

//mode control register bits
#define ON_OFF			5
#define MASTER_SLAVE	4 //1=maser 0=slave
#define BLINK			3
#define CURSOR			2
#define GRAFIC			1
#define EXT_CG			0 //0=internal char generation

#define HD61830_DELAY	asm volatile ("nop\n nop\n nop\n nop\n");
unsigned short hd61830Addr = 0; // current address counter

void hd61830BusyCheck(void)
{
	RS_HIGH;
	RW_HIGH;
	E_HIGH;
	DATA_INPUT;
	HD61830_DATA_PORT=0;
	while (HD61830_DATA_PIN & 0x80)
	{ 
		E_LOW;
		HD61830_DELAY;
		E_HIGH;
		HD61830_DELAY;
	}
	E_LOW;
}

void hd61830Command(unsigned char reg, unsigned char data)
{
	hd61830BusyCheck();
	E_HIGH;
	RS_HIGH;
	RW_LOW;
	DATA_OUTPUT;
	HD61830_DATA_PORT=reg;
	E_LOW;
	HD61830_DELAY;
	E_HIGH;
	RS_LOW;
	RW_LOW;
	DATA_OUTPUT;
	HD61830_DATA_PORT=data;
	E_LOW;
}

unsigned char hd61830Read(unsigned char reg)
{
	unsigned char result;
	hd61830BusyCheck();
	E_HIGH;
	RS_HIGH;
	RW_LOW;
	DATA_OUTPUT;
	HD61830_DATA_PORT=reg;
	E_LOW;
	HD61830_DELAY;
	E_HIGH;
	RS_LOW;
	RW_HIGH;
	DATA_INPUT;
	result = HD61830_DATA_PORT;
	E_LOW;
	return result;
}

void hd61830Init(void)
{
	HD61830_DATA_PORT= 0; // alle null
	HD61830_DATA_DDR = 0; // alle input
	sbi(HD61830_CONTROL_DDR,	HD61830_RS);
	sbi(HD61830_CONTROL_DDR,	HD61830_RW);
	sbi(HD61830_CONTROL_DDR,	HD61830_E);
	RS_LOW;
	RW_LOW;
	E_LOW;
	#ifdef HD61830_CS
	sbi(HD61830_CONTROL_DDR,	HD61830_CS);
	CS_HIGH;
	#endif
	hd61830Command(MODE,(1<<ON_OFF)|(1<<MASTER_SLAVE)|(1<<BLINK)|(1<<CURSOR)|(0<<EXT_CG)|(0<<GRAFIC));
	hd61830Command(CHAR_PITCH, HD61830_FONT_Y<<4|HD61830_FONT_X); // set 5x7 font
	hd61830Command(NUM_OF_CHAR, HD61830_CHARS_LINE - 1); // 
	hd61830Command(NUM_OF_TIME_DIVISIONS, HD61830_YPIXELS - 1); //?
	hd61830Command(CURSOR_POS,HD61830_FONT_Y);
	hd61830Command(START_LOW_ADDR, 0);
	hd61830Command(START_HIGH_ADDR, 0);
	hd61830Command(LOW_ADDR, 0);
	hd61830Command(HIGH_ADDR, 0);
	hd61830Clear();
}
// moves the cursor/position to Home (upper left corner)
void hd61830Home(void)
{
	hd61830Command(LOW_ADDR, 0);
	hd61830Command(HIGH_ADDR, 0);
}

// clears the LCD display
void hd61830Clear(void)
{
	unsigned short i;
	hd61830Command(LOW_ADDR, 0);
	hd61830Command(HIGH_ADDR, 0);
	for(i=0 ;i<HD61830_YPIXELS*HD61830_CHARS_LINE; i++) hd61830Command(WRITE_DATA, 0);
	hd61830Command(LOW_ADDR, 0);
	hd61830Command(HIGH_ADDR, 0);
}

// prints a series of bytes/characters to the display
void hd61830PrintData(char* data, unsigned char nBytes)
{
	unsigned char i;

	// check to make sure we have a good pointer
	if (!data) return;

	// print data
	for(i=0; i<nBytes; i++)
	{
		hd61830Command(WRITE_DATA, data[i]);
	}
}

void hd61830GotoXY(unsigned char x, unsigned char y)
{
	unsigned short addr = x+y*HD61830_CHARS_LINE;
	hd61830Command(LOW_ADDR, addr);
	hd61830Command(HIGH_ADDR, addr>>8);
}
