
//*****************************************************************************
//
// File Name	: 'uart.c'
// Title			: UART driver with buffer support
// Author		: Pascal Stang - Copyright (C) 2000-2002
// Created		: 11/22/2000
// Revised		: 11/01/2001
// Version		: 1.3
// Target MCU	: ATMEL AVR Series
// Editor Tabs	: 3
//
// This code is distributed under the GNU Public License
//		which can be found at http://www.gnu.org/licenses/gpl.txt
//
//*****************************************************************************

#include <avr/io.h> 
#include <avr/interrupt.h>
#include <avr/signal.h>
#include <avr/pgmspace.h>

#include "global.h"
//#include "strings.h"
#include "buffer.h"
#include "uart.h"
 
// UART global variables
// flag variables
volatile u08   UART_ReadyTx;
volatile u08   UART_BufferedTx;
// receive and transmit buffers
cBuffer UART_RxBuffer;
cBuffer UART_TxBuffer;
unsigned short UART_RxOverflow;
#ifndef UART_BUFFERS_EXTERNAL_RAM
	// using internal ram,
	// automatically allocate space in ram for each buffer
	static char UART_RxData[UART_RX_BUFFER_SIZE];
	static char UART_TxData[UART_TX_BUFFER_SIZE];
#endif

void UART_Init(void)
{
	// initialize the buffers
	UART_InitBuffers();

	// enable RxD/TxD and interrupts
	#ifdef UCR
		// this line for AT90S8515,8535,ATmega103,etc
		outb(UCR, BV(RXCIE)|BV(TXCIE)|BV(RXEN)|BV(TXEN));
	#endif
	#ifdef UCSRB
		// this line for the Mega163
		outb(UCSRB, BV(RXCIE)|BV(TXCIE)|BV(RXEN)|BV(TXEN));
	#endif
	#ifdef UCSR0B
		// this line for the Mega161/Mega128
		outb(UCSR0B, BV(RXCIE)|BV(TXCIE)|BV(RXEN)|BV(TXEN));
	#endif

	// set default baud rate
	UART_SetBaudRate(UART_BAUD_RATE);  

//        outp( (u08)(1666>>8), UBRRH);
//        outp( (u08)1666, UBRRL);

	// initialize states
	UART_ReadyTx = TRUE;
	UART_BufferedTx = FALSE;
	// clear overflow count
	UART_RxOverflow = 0;
	// enable interrupts
	sei();
}

void UART_InitBuffers(void)
{
	#ifndef UART_BUFFERS_EXTERNAL_RAM
		// initialize the UART receive buffer
		bufferInit(&UART_RxBuffer, UART_RxData, UART_RX_BUFFER_SIZE);
		// initialize the UART transmit buffer
		bufferInit(&UART_TxBuffer, UART_TxData, UART_TX_BUFFER_SIZE);
	#else
		// initialize the UART receive buffer
		bufferInit(&UART_RxBuffer, (u08*) UART_RX_BUFFER_ADDR, UART_RX_BUFFER_SIZE);
		// initialize the UART transmit buffer
		bufferInit(&UART_TxBuffer, (u08*) UART_TX_BUFFER_ADDR, UART_TX_BUFFER_SIZE);
	#endif
}

void UART_SetBaudRate(u16 baudrate)
{
	// calculate division factor for requested baud rate, and set it
	outb(UBRRL, (u08)((F_CPU+(baudrate*8L))/(baudrate*16L)-1));
}

u16 UART_GetDataLength(void)
{
	return UART_TxBuffer.datalength;
}

cBuffer* UART_GetRxBuffer(void)
{
	// return rx buffer pointer
	return &UART_RxBuffer;
}

cBuffer* UART_GetTxBuffer(void)
{
	// return tx buffer pointer
	return &UART_TxBuffer;
}

void UART_SendByte(u08 txData)
{
   UART_SendBuffer(&txData,1);
/*
	// wait for the transmitter to be ready
	while(!UART_ReadyTx);
	// send byte
	outp( txData, UDR );
	// set ready state to FALSE
	UART_ReadyTx = FALSE;
*/
}

u08 UART_ReceiveByte(void)
{
	// make sure we have a receive buffer
	if(UART_RxBuffer.size)
	{
		// make sure we have data
		if(UART_RxBuffer.datalength)
		{
			// get byte from beginning of buffer
			return bufferGetFromFront(&UART_RxBuffer);
		}
		else
		{
			// no data
			return 0;
		}
	}
	else
	{
		// no buffer
		return 0;
	}
}

void UART_FlushReceiveBuffer(void)
{
	// flush all data from receive buffer
	bufferFlush(&UART_RxBuffer);
	// same effect as above
	// uartRxBuffer.datalength = 0;
}

u08 UART_ReceiveBufferIsEmpty(void)
{
	if(UART_RxBuffer.datalength == 0)
	{
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}
/*
void UART_SendTxBuffer(void)
{
	// turn on buffered transmit
	UART_BufferedTx = TRUE;
	// send the first byte to get things going by interrupts
	UART_SendByte(bufferGetFromFront(&UART_TxBuffer));
}
*/

u08 UART_SendBuffer(char *buffer, u16 nBytes)
{
//	register u08 first;
	register u16 i;

	// check if there's space (and that we have any bytes to send at all)
	if((UART_TxBuffer.datalength + nBytes < UART_TxBuffer.size) && nBytes)
	{
		// grab first character
//		first = *buffer++;
		// copy user buffer to uart transmit buffer
		for(i = 0; i < nBytes; i++)
		{
			// put data bytes at end of buffer
			bufferAddToEnd(&UART_TxBuffer, *buffer++);
		}

		// send the first byte to get things going by interrupts
		UART_BufferedTx = TRUE;
//		UART_SendByte(first);
		if(UART_ReadyTx){
			// send byte
			outp( bufferGetFromFront(&UART_TxBuffer), UDR );
			// set ready state to FALSE
			UART_ReadyTx = FALSE;
		}
		// return success
		return TRUE;
	}
	else
	{
		// return failure
		return FALSE;
	}
}

void UART_PrintfProgStr(u08* pBuf)
{
  for(;PRG_RDB(pBuf)!=0;UART_SendByte(PRG_RDB(pBuf++)));
}

void UART_PrintStr(u08 s[])
{
  u08 i;
  for(i=0;s[i]!=0;i++) UART_SendByte(s[i]);
}

void UART_PrintfEndOfLine(void)
{
  UART_SendByte(13);
  UART_SendByte(10);
}

// UART Transmit Complete Interrupt Function
SIGNAL(SIG_UART_TRANS)
{
	// check if buffered tx is enabled
	if(UART_BufferedTx)
	{
		// check if there's data left in the buffer
		if(UART_TxBuffer.datalength)
		{
			// send byte from top of buffer
			outp( bufferGetFromFront(&UART_TxBuffer), UDR );
		}
		else
		{
			// no data left
			UART_BufferedTx = FALSE;
			// return to ready state
			UART_ReadyTx = TRUE;
		}
	}
	else
	{
		// we're using single-byte tx mode
		// indicate transmit complete, back to ready
		UART_ReadyTx = TRUE;
	}
}

// UART Receive Complete Interrupt Function
SIGNAL(SIG_UART_RECV)
{
	// put received char in buffer
	// check if there's space
	if( !bufferAddToEnd(&UART_RxBuffer, inp(UDR)) )
	{
		// no space in buffer
		// count overflow
		UART_RxOverflow++;
	}
}
